//___________________________________
//                                   \
// Plethora Event algorithms for cutscenes. version 1.6 	$Id: plethora.js 119 2008-07-28 22:34:44Z nilton $
//___________________________________/
//
/*
	Usage:
	Declare this once as a global variable:
		var EventQueue = new PlethoraEngine();

	Also, in your RenderScript (See SetRenderScript in the Sphere Documentation), you run:
		EventQueue.UpdateAndDraw();

	Then, when starting a cutscene, you can initialize it (or abort a running cutscene) like this:

		EventQueue.reset();

	This will clear and pause the cutscene. But when any cutscene ends, 
	it will be in that state, so it's only if you need to clear/cancel another cutscene.
	(Which is usually not the case and is probably a bad idea)

	You can add items like so:
		EventQueue.add( $myStringWithCommands ); // A string
		EventQueue.add( function(){my function goes here} ); // An unnamed function
		EventQueue.add( function_name,null,[param1,param2...] ); // a function with parameters (see that null, that's the context, important)
		EventQueue.delayms( 1000 ); // Wait 1 second (you also have .delayFrames(); )
		EventQueue.addAndDelay(1000,$myStringWithCommands); // Same as 2 cmd's above, bundled, it first executes the command, then the delay
		EventQueue.add($myStringWithCommands).delay(1000); // Same as cmd above
		EventQueue.wait( $myBooleanString ); // Waits until the boolean expression is true. (transforms internally to an efficient function)

	And even like so:
		function jump(sprite,direction){}; // Make this function do something interesting :P
		EventQueue.RegisterAction(jump, ['sprite','direction'], {sprite:GetInputPerson(), wait:true}, 'JUMP'); // Register action only once
		EventQueue.add( {action:'JUMP', sprite:"hero", direction: 'north' } ); // action can now be used multiple times


	Finally, you call EventQueue.play(); to start playback 

	Dont call EventQueue.run() yourself, use EventQueue.UpdateAndDraw() inside a RenderScript.
	For example: SetRenderScript( "EventQueue.UpdateAndDraw()" );
	When no event is running, EventQueue.UpdateAndDraw(); is still very efficient to run without slowdown.
	If the MapEngine is not running, you can also use EventQueue.autoplay()

	You can short-write some functions (Make function call smaller. (just like sin=Math.sin))
	But nQ=EventQueue.add; doesnt work, because it doesnt expand the "this." inside add()
	this is the way how to do it:
		nQ = function(event){ EventQueue.add.apply(EventQueue,arguments); }
		uQ = function(event){ EventQueue.unshift.apply(EventQueue,arguments); } // maybe not required
		tQ = function(delaymsec,event){ EventQueue.addAndDelay.apply(EventQueue,arguments); }
		fQ = function(delayframes,event){ EventQueue.addAndDelayFrames.apply(EventQueue,arguments); }
		wQ = function(bstr, unshift){ EventQueue.wait(bstr, unshift); }
		dQ = function(ms, unshift){ EventQueue.delayms(ms, unshift); }
		zQ = function(frames, unshift){ EventQueue.delayFrames(frames, unshift); }

	
	Some other handy functions:
		// Waits until a sound_object has reached a certain position
		soundPosWaitQ = function(snd,pos, unshift){ EventQueue.wait(snd+".getPosition()>"+pos,unshift); }

		// Enable, Disable or delay FlipScreen() for EventQueue.autoplay(); (See {@link PlethoraEngine#SetFlipScreen})
		FlipScreenQ = function(onoff) { nQ(Game.EventQueue.SetFlipScreen, Game.EventQueue, onoff); }

	some notes on EventQueue.add() with functions:

	It is ok to queue Global sphere functions, but you can't queue functions inside native objects, so this is legal:
		nQ(CreatePerson, null, 'hero', 'hero.rss', true); // A global function
		nQ(Lithonite.NPCSequence, Lithonite, who, dir); // A function inside a custom object, in the context of that object
	But this is not:
		nQ(img.blit, img, 0, 0); // A function inside a primitive
	It has to be rewritten as a string (or be inside an unnamed function): 
		nQ( 'img.blit(0,0)' ); // A string that will be evaluated

*/

/**
 * Create a new PlethoraEngine object, a cutscene event handler.
 * @constructor
 * @param {object} init Optional object with variable defaults (see example below)
 * @return A new PlethoraEngine Object
 * example:
 *   Scene = new PlethoraEngine( { strictparams: true, fps: 100 } );
 */
function PlethoraEngine(init)
{
	if (this instanceof PlethoraEngine == false) 
		return new PlethoraEngine(init);
	this.Queue = new Array(); // Will hold our 'events'
	this.waiting = false; // Are we waiting for a boolean expression to be true?
	this.active = false; // Is UpdateAndDraw() doing something?
	this.name = ""; // Name of the playing cutscene
	this.UpdateAndDraw = function(){}; // This runs in your renderscript, it changes when events are running, and returns to this when done.
	this.frames = 0; // Internal counter for delayFrames(), you can keep track of when the EventQueue is going to continue again.
	this.time= 0; // Internal counter for delayms(), you can keep track of when the EventQueue is going to continue again.
	this.fps = 60; // Default frames per second when .autoplay() is selected
	this.fsf = true; // FlipScreen enabled for .autoplay()
	this.event = undefined; // current eventline executing.
	this.action = new Object(); // will hold defined actions
	this.scenelet = new Object(); // will hold defined scenelet
	this.cancancel = false; // Set to true if IsAnyKeyPressed() is tested to cancel the cutscene
	this.strictparams = false; // Set to true to check redundant parameters of actions and scenelets

	for (action in init){
		if( (typeof this[action] == 'function') && (typeof init[action] == 'object') )
			this[action].apply(this, init[action]);
		else
			this[action] = init[action];
	}
}

/**
 * Reset the PlethoraEngine
 * @param {Boolean} all True if the actions defined with RegisterAction() also have to be cleared
 */
PlethoraEngine.prototype.reset = function(all){
	this.Queue = [];
	this.UpdateAndDraw = function(){};
	this.active = false;
	this.waiting = false;
	this.name = "";
	this.fsf = true;
	this.cancancel = false;
	this.event = undefined;
	if(all)
		this.action = new Object(); // will hold defined actions
	GarbageCollect();
	return this;
}

/**
 * Eats up the first element(s) of an arguments object. ( like arguments.shift(); )
 * @param {array} arg Array Object
 * @param {integer} n How many placers to shift, 1 by default.
 * @returns An array without the first element of the given argument array
 */
PlethoraEngine.prototype.shiftArgs = function(arg,n){
	var params = new Array();
	for (var i = n||1, len = arg.length; i < len; i++) {
		params.push(arg[i]);
	}
	return params;
}

/**
 * Add a new event to the Queue, the events will be played back in the same order.
 * @param {blob} event Event to queue. It can be a string, a function with optional parameters, or an action object.
 * @param {this} context If 'event' is a function inside an object, you need to give it 'this', example:
 * EventQueue.add( Lithonite.repos, Lithonite, 'hero');
 */
PlethoraEngine.prototype.add = function(event, context){
	if(typeof event == 'function'){
		this.Queue.push({func: event, obj:context||null, params: this.shiftArgs(arguments,2)});
	} else {
		this.Queue.push(event);
	};
	return this;
}

/**
 * Similar to add(), but it will add the event to the beginning of the (running) Queue.
 * more about unshift: It's used when already running the queue, and you are just executing something that
 * at that moment decides that the queue should pause a little for it to finish execution. 
 * multiple unshift activities have to be added in reversed order.
 */
PlethoraEngine.prototype.unshift = function(event, context){
	if(typeof event == 'function'){
		this.Queue.unshift({func: event, obj:context||null, params: this.shiftArgs(arguments,2)});
	} else {
		this.Queue.unshift(event);
	};
	return this;
}


/**
 * Same as add, but also delays afterwards for delaymsec milliseconds.
 * @param {integer} delaymsec Milliseconds to wait before continuing with the next event
 * @param {blob} event Event to queue. It can be a string, a function with optional parameters, or an action object.
 */
PlethoraEngine.prototype.addAndDelay = function(delaymsec,event){
	this.add.apply(this, this.shiftArgs(arguments));
	this.delayms(delaymsec);
	return this;
}

/**
 * Same as add, but also delays afterwards for delayframes frames.
 * param {integer} delayframes Frames to wait before continuing with the next event
 * @param {blob} event Event to queue. It can be a string, a function with optional parameters, or an action object.
 */
PlethoraEngine.prototype.addAndDelayFrames = function(delayframes,event){
	this.add.apply(this, this.shiftArgs(arguments));
	this.delayFrames(delayframes);
	return this;
}

/**
 * Internal function, handles errors
 */
PlethoraEngine.prototype.HandleError = function(msg,e){
	for (var i in e) msg += i + ' = ' + e[i] +"\n";
	if(this.name) msg = "While plethora executed cutscene '"+this.name+"':\n" + msg;
	Abort(msg);
}

/**
 * Internal function, executed by {@link PlethoraEngine#UpdateAndDraw}
 * strings are evalled, functions have been wrapped into an object.
 */
PlethoraEngine.prototype.run=function(){
	if(this.waiting) return;
	if(this.Queue.length==0){
		this.reset();
		this.fsf = false;
		return;
	}

	this.event = this.Queue.shift();

	if(typeof this.event =='string') {
		try{
			eval(this.event); 
		} catch (e) {
			 this.HandleError("Plethora eval error: "+this.event+"\n", e);
		}
	}else if(typeof this.event =='object'){
		// Check if func or action
		if(this.event.func) {
			this.event.func.apply(this.event.obj||null, this.event.params);
		}else if (this.event.action){
			if(!this.action[this.event.action]) {
				this.HandleError('FATAL: Plethora received an undefined action: '+this.event.action);
				return;
			}
			// Check typo's in input
			if(this.strictparams){
				var paramlist = new Object();
				for(var i=0, l=this.action[this.event.action].order.length ; i<l ; ++i){
					paramlist[this.action[this.event.action].order[i]] = i;
				}
				for(var i in this.event){
					if(i == 'action') continue;
					if(!(i in paramlist)){
						this.HandleError('FATAL: Plethora received an unknown parameter "'
						+ i
						+'" in action: '+this.event.action);
					}
				}
			}
			// Loop through default values, set them where needed
			for(var i in this.action[this.event.action].defaults){
				if(this.event[i] === undefined)
					this.event[i] = this.action[this.event.action].defaults[i];
			}
			// Now check for mandatory parameters
			if(this.action[this.event.action].mandatory){
				for(var i=0, len=this.action[this.event.action].mandatory.length ; i<len ; i++){
					if(!(this.action[this.event.action].mandatory[i] in this.event)){
						this.HandleError('FATAL: Plethora misses a parameter "'
						+ this.action[this.event.action].order[i]
						+ '" for action: '+this.event.action);
					}
				}
			}
			// fill the parameter array in the correct order
			// There are 2 cases, order is given as an array, or as an object
			var params=[];
			if(this.action[this.event.action].order[0]){
				for(var i=0, len=this.action[this.event.action].order.length ; i<len ; i++){
					params[i] = this.event[this.action[this.event.action].order[i]];
				}
			}else{
				for(var i in this.action[this.event.action].order){
					params[ this.action[this.event.action].order[i] ] = this.event[i];
				}
			}
			// Finally execute the action
			this.active = this.event.action; // If the scene kills Sphere, at least know which one.
			try{
				this.action[this.event.action].func.apply(this.action[this.event.action].obj,params);
			} catch (e) {
			 	this.HandleError("Plethora action error: "+this.active+"\n", e);
			}
			this.active = true;
		}else if (this.event.scenelet){
			if(!this.scenelet[this.event.scenelet]) {
				this.HandleError('FATAL: Plethora received an undefined scenelet: '+this.event.scenelet);
				return;
			}
			// Check typo's in input
			if(this.strictparams){
				var paramlist = new Object();
				for(var i=0, l=this.scenelet[this.event.scenelet].order.length ; i<l ; ++i){ 
					paramlist[this.scenelet[this.event.scenelet].order[i]] = i;
				}
				for(var i in this.event){
					if(i == 'scenelet') continue;
					if(!(i in paramlist)){
						this.HandleError('FATAL: Plethora received an unknown parameter "'
						+ i
						+'" in scenelet: '+this.event.scenelet +FoldObj(this.scenelet[this.event.scenelet]));
					}
				}
			}
			// Loop through default values, set them where needed
			for(var i in this.scenelet[this.event.scenelet].defaults){
				if(this.event[i] === undefined)
					this.event[i] = this.scenelet[this.event.scenelet].defaults[i];
			}
			// Now check for mandatory parameters
			if(this.scenelet[this.event.scenelet].mandatory){
				for(var i=0, len=this.scenelet[this.event.scenelet].mandatory.length ; i<len ; i++){
					if(!(this.scenelet[this.event.scenelet].mandatory[i] in this.event)){
						this.HandleError('FATAL: Plethora misses a parameter "'
						+ this.scenelet[this.event.scenelet].order[i]
						+ '" for scenelet: '+this.event.scenelet);
					}
				}
			}
			// fill the parameter array in the correct order
			// There are 2 cases, order is given as an array, or as an object
			var params=[];
			if(this.scenelet[this.event.scenelet].order[0]){
				for(var i=0, len=this.scenelet[this.event.scenelet].order.length ; i<len ; i++){
					params[i] = this.event[this.scenelet[this.event.scenelet].order[i]];
				}
			}else{
				for(var i in this.scenelet[this.event.scenelet].order){
					params[ this.scenelet[this.event.scenelet].order[i] ] = this.event[i];
				}
			}
			// Finally execute the scenelet
			this.active = this.event.scenelet; // If the scene kills Sphere, at least know which one.
			try{
				this.scenelet[this.event.scenelet].init.apply(this.scenelet[this.event.scenelet],params);
			} catch (e) {
				this.HandleError("Plethora scenelet init error: "+this.active+"\n", e);
			}
			this.SetFlipScreen(1); // Do not FlipScreen after initializing
			this.UpdateAndDraw = Function(
				"try{"+
				"if(" +
					"this.scenelet['"+this.event.scenelet+"'].update()" +
				"){" + 
					"this.scenelet['"+this.event.scenelet+"'].done();" +
					"this.play(true);"+
				"};"
				+"}catch (e){this.HandleError('Plethora scenelet update error: "+this.active+"'+'\\n',e)};"
			);
		}else{
			this.HandleError('FATAL: Plethora received an object without action string: '+FoldObj(this.event));
			//Unknown object, silently continue
		};
	}else if(typeof this.event =='function'){
		try{this.event();}catch (e){this.HandleError("Plethora function error: \n", e);}
	}else {
		try{eval(this.event)}catch (e){this.HandleError("Plethora eval2 error: \n", e);}
	}
}

/**
 * Define a function with default parameters to be queued as a verbose object
 * @param {value} alias How to call this function with {@link PlethoraEngine#add}. Can be a String or Integer/Constant
 * @param {Object} context Base Object for the function, set to null if its a global function.
 * @param {Function} ACTION The name of a function without the ().
 * @param {Array} paramorder Order of the parameters that have to be supplied to the ACTION function
 * @param {Object} defaultparam Set the default parameters, so you dont have to define them all each time. parameters that are in paramorder but not here are mandatory.
 * @param {Array} mandatoryparam Array with mandatory parameters that must be supplied. (Will throw an error if one of these parameters is not supplied)
 * example:
 *	 function jump(sprite,direction){}; // This the function we want to register
 *	 EventQueue.RegisterAction(null,jump, ['sprite','direction'], {sprite:GetInputPerson(), wait:true}, 'JUMP');
 *	 EventQueue.add( {action:'JUMP', sprite:"hero" } );
 *
 * In this example, we need to set the context because the function we register has 'this.' inside it:
 *	 Game.F = new Object();
 *	 Game.F.sum = function(a,b) {this.result=a+b; return this.result;}
 *	 EventQueue.RegisterAction(Game.F, Game.F.sum, ['x', 'y'], {}, 'sum');
 *	
 * Another example:
 *	Game.EventQueue.RegisterAction(
 *		'Emote',
 *		null,
 *		function(sprite,emotion,wait){
 *			var emote = Game.lithonite.EMOTIONS[emotion](sprite);
 *			if (wait)
 *				zQ( Game.lithonite.getSeqTimes(emote, emotion) ,true); // Prepend a pause at the beginning of the running eventqueue
 *		},
 *		['sprite', 'emotion', 'wait'],
 *		{sprite: Game.lithonite.GIP, wait:true},
 *		['emotion']
 *	);
 * Which we run like this:
 * 	nQ( { action: 'Emote', emotion: 'heart' } );
 */
PlethoraEngine.prototype.RegisterAction = function(alias, context, ACTION, paramorder, defaultparam, mandatoryparam){
	this.action[alias||ACTION]={
		func: ACTION,
		order: paramorder,
		defaults: defaultparam,
		mandatory: mandatoryparam,
		obj: context||null
	}
	return this;
}

/**
 * Copied from VvB: Scenelets, 
 * @param {value} alias How to call this function with {@link PlethoraEngine#add}. Can be a String or Integer/Constant
 * @param {Function} init This function receives the initializing parameters and does something with them (use this. notation)
 * @param {Function} update This function keeps running each frame, when it finally returns true, its done.
 * @param {Function} done This function is to cleanup, usually its just empty, like this:	done: function(){}
 * @param {Array} paramorder Order of the parameters that have to be supplied to the init function
 * @param {Object} defaultparam Set the default parameters, so you dont have to define them all each time.
 * @param {Array} mandatoryparam Array with mandatory parameters that must be supplied. (Will throw an error if one of these parameters is not supplied)
 * example:
 *	EventQueue.RegisterScenelet(
 *		'Countdown',
 *		function(f,t){ this.from = f; this.to = t; },
 *		function(){ if(this.from == this.to) return true; GetSystemFont().drawText(20,10, " SCENELET i="+this.from); --this.from; },
 *		function(){},
 *		['startingnumber', 'endingnumber'],
 *		{startingnumber: 5, endingnumber: 0},
 *              [],
 *	);
 *
 *  nQ( { scenelet: 'Countdown', from: 15, to: 1 } );
 *
 * note: the information stored in this.XXX variables is persistent, for example, we can read the 'to' variable like this:
 *	EventQueue.scenelet['Countdown'].to
 * That is why you need to be careful to reset the values in init().
 */
PlethoraEngine.prototype.RegisterScenelet = function(alias, init, update, done, paramorder, defaultparam, mandatoryparam){
	this.scenelet[alias]={
		init: init,
		update: update,
		done: done,
		order: paramorder,
		mandatory: mandatoryparam,
		defaults: defaultparam,
	}
	return this;
}

/**
 * Delay the execution of the EventQueue for 'frames' frames.
 * @param {integer} frames Number of frames to delay, range: 1 and up.
 * @param {Boolean} unshift If we need to add a delay at the beginning (of things currently executing), not at the very end
 */
PlethoraEngine.prototype.delayFrames = function(frames, unshift){
	if(frames<1) frames = 1;
	if(unshift){
		this.wait("--this.frames==0", unshift);
		this.unshift("this.frames="+frames);
	}else{
		this.add("this.frames="+frames);
		this.wait("--this.frames==0");
	}
	return this;
}

/**
 * Delay the execution of the EventQueue for 'milliseconds' milliseconds.
 * @param {integer} milliseconds Number of milliseconds to delay
 * @param {Boolean} unshift If we need to add a delay at the beginning, not at the end
 */
PlethoraEngine.prototype.delayms=function(milliseconds, unshift){
	if(unshift){
		this.wait("GetTime()>this.time", unshift);
		this.unshift("this.fsf=false; this.time=GetTime()+" + milliseconds);
	}else{
		this.add("this.fsf=false; this.time=GetTime()+" + milliseconds);
		this.wait("GetTime()>this.time");
	}
	return this;
}

/**
 * Start executing the EventQueue. It also continues the queue when paused or waiting.
 * You need to have UpdateAndDraw() defined in a renderscript for it to work.
 * @param {string} CutSceneName Name of the cutscene you are playing. Optional. Use 'true' to continue a pause and preserve the cutscene name.
 */
PlethoraEngine.prototype.play = function(CutSceneName) {
	this.waiting = false;
	this.active = true; 
	this.UpdateAndDraw = this.run;
	if(CutSceneName !== true){
		this.name = CutSceneName;
		this.fsf = true;
	}else if (!this.fsf)
		this.fsf = true;
}

/**
 * Start executing the EventQueue. It also continues the queue when paused or waiting.
 * This version if {@link PlethoraEngine#play} does not depend on a running mapengine on a renderscript or mapengine to work.
 * @param {string} CutSceneName Name of the cutscene you are playing. Optional
 * @param {integer} fps Default frames per second between FlipScreens
 * @param {Boolean} cancancel Set if the cutscene can be aborted by pressing any key.
 * @returns true if cancelled ( EventQueue.cancancel must be true for this to work)
 */
PlethoraEngine.prototype.autoplay = function(CutSceneName, fps, cancancel) {
	this.play(CutSceneName);

	fps = fps || this.fps || 60;
	var delay = Math.floor(1000 / fps);
	var cont; // When to continue with next frame
	if(cancancel !== undefined){
		this.cancancel = cancancel;
	}
	while(this.active){
		this.preautoplay();
		this.UpdateAndDraw();
		if(typeof this.fsf == 'boolean'){
			if(this.fsf)
				FlipScreen();
		}else if (--this.fsf<=0){
			this.fsf = true;
		}
		this.postautoplay();
		cont = GetTime() + delay;
		if(this.cancancel && IsAnyKeyPressed()) {
			this.reset();
			return false;
		}
		while (GetTime()<cont) { this.whileautoplay(); };
	}
	this.reset();
	return true;
}

/*
 * These 3 functions do things before autoplay draws, after and while it waits for the next update.
 * You can redefine them at will
 */
PlethoraEngine.prototype.preautoplay = function() {};
PlethoraEngine.prototype.postautoplay = function() {};
PlethoraEngine.prototype.whileautoplay = function() {};

/** 
 * Disables/enables or delays  FlipScreen() for autoplay,
 * @param {integer} frames Boolean to enable/disable, a number >0 to delay n queued commands
 */
PlethoraEngine.prototype.SetFlipScreen = function(frames){
	this.fsf = frames;
	return this;
}

/**
 * Stop executing the EventQueue, use {@link PlethoraEngine#play} to start it again
 */
PlethoraEngine.prototype.pause = function() {
	this.waiting = true;
	this.active = true; 
	this.UpdateAndDraw = function(){}; 
	return this;
}

/**
 * Stop executing the EventQueue, until the boolean expression 'boolstr' is true.
 * @param {Boolean} boolstr A boolean string
 * @param {Boolean} unshift If we need to add a delay at the beginning, not at the end
 * This actually modifies UpdateAndDraw() to a function that runs the 'boolstr' until it is true, then it starts the queue again.
 */
PlethoraEngine.prototype.wait = function(boolstr,unshift) {
	var waitfunc = Function(
		"this.fsf=false;" +
		"this.waiting=true;" +
		"this.UpdateAndDraw=function(){if(" + boolstr + "){this.play(true); this.fsf=1;}}" );
	if(unshift)
		this.unshift(waitfunc, this);
	else
		this.add(waitfunc, this);
	return this;
}

